/** @file   mp3player.h
 * @brief   Declaration of MP3Player class
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:38 $
 * @author  Tomi Lamminsaari
 */


#ifndef H_ENG2D_MP3PLAYER_H
#define H_ENG2D_MP3PLAYER_H

#include <allegro.h>
#include "eng2d_dll.h"
#include "mp3file.h"


namespace eng2d {


/** @class  MP3Player
 * @brief   Plays the MP3File
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 *
 * This class has only static members/methods since only one mp3-file can
 * be played at a time.
 *
 * Use the <code> setMP3( MP3File* ) </code> - method to set the mp3 file.
 * Then call <code> play() </code> - method and after that you should keep
 * calling the <code> poll() </code> method about 35 times per second to
 * get smooth playpack. Of course this polling interval depends on the
 * value of <code> MP3Player::bufferSize </code> member.
 *
 */
class DLLIMPORT MP3Player
{
public:

  ///
  /// Static members and methods
  /// ==========================
  
  /** Are we currently playing a mp3-file. */
  static bool      playing;
  /** Size of the decode-buffer. */
  static long      bufferSize;
  /** Current playing volume. */
  static int       vol;
  /** Current pan-value. 128 = center, 0=left, 255=right */
  static int       pan;
  /** Pointer to the mp3-file we're playing. */
  static MP3File*  pMusic;


  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructor. NOT IMPLEMENTED!
   */
  MP3Player();
  
  /** Destructor. NOT IMPLEMENTED!
   */
  ~MP3Player();
  

  ///
  /// Methods
  /// =======

  /** Sets the mp3-file you want to play next.
   * @param     pMP3              Pointer to new mp3-file or 0 pointer if
   *                              you just want to close the current mp3-file.
   * @return    'true' if failed to set new mp3-file.
   */
  static bool setMP3( MP3File* pMP3 );
  
  /** Sets the size of the decode-buffer.
   * @param     buffsize          New buffersize.
   */
  static void setBufferSize( long buffsize );
  
  /** Starts or continues playing the current mp3-file.
   * @return    ALMP3_OK if successful.
   * @see       AllegroMP3-documentation for more information about the possible
   *            return-values.
   */
  static int play();
  
  /** Starts or continues playing the current mp3-file.
   * @param     vol               The playback volume.
   * @param     pan               The pan position.
   * @return    ALMP3_OK if successful.
   * @see       AllegroMP3-documentation for more information about the
   *            possible return-values.
   */
  static int play(int vol, int pan);
  
  /** Stops (actually pauses) the playback.
   */
  static void stop();
  
  /** Decodes the mp3-data to the playback buffer.
   * <p>
   * This method must be called within regular intervals to keep the mp3-file
   * playing properly. About 40 times a second seems to be enough.
   * @return    ALMP3_OK if everything went alright.
   *            ALMP3_POLL_PLAYJUSTFINISHED if we just decoded the last frame
   *            of mp3-file.
   * @see       AllegroMP3-documentation for more information about the
   *            return-values the 'poll()'-method can have.
   */
  static int poll();
  
  /** Sets the playback volume.
   * @param     v                 New volume.
   */
  static void volume(int v);
  
  /** Sets the pan-position.
   * @param     p                 New pan-position.
   */
  static void panning(int p);
  

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the current playback volume.
   * @return    Current playback volume.
   */
  static int volume();
  
  /** Returns the current pan-position.
   * @return    Current pan-position
   */
  static int panning();
  

private:

  /** Private methods **/

  MP3Player(const MP3Player& rO);
  MP3Player& operator = (const MP3Player& rO);


  /** Members **/
  
  
};

};

#endif
